const darkModeMediaQuery = window.matchMedia('(prefers-color-scheme: dark)');
const notForced = () => !('colorSchemeOs' in document.documentElement.dataset);
const lightColor = 'rgba(255, 255, 255, 0.8)';
const darkColor = 'rgba(0, 0, 0, 0.8)';
const getColorScheme = () => {
  if (notForced()) {
    return darkModeMediaQuery.matches ? darkColor : lightColor;
  }
  if ('colorScheme' in document.documentElement.dataset) {
    return document.documentElement.dataset.colorScheme === 'dark' ? darkColor : lightColor;
  }
  return darkModeMediaQuery.matches ? darkColor : lightColor;
};

/**
 * Creates a custom element with the default spinner of the Joomla logo
 */
class JoomlaCoreLoader extends HTMLElement {
  get inline() {
    return this.hasAttribute('inline');
  }
  set inline(value) {
    if (value !== null) {
      this.setAttribute('inline', '');
    } else {
      this.removeAttribute('inline');
    }
  }
  get size() {
    return this.getAttribute('size') || '345';
  }
  set size(value) {
    this.setAttribute('size', value);
  }
  get color() {
    return this.getAttribute('color');
  }
  set color(value) {
    this.setAttribute('color', value);
  }
  static get observedAttributes() {
    return ['color', 'size', 'inline'];
  }
  constructor() {
    super();
    this.attachShadow({ mode: 'open' });
    const template = document.createElement('template');
    template.innerHTML = `
      <style>
        :host {
          z-index: 10000;
          justify-content: center;
          align-items: center;
          display: flex;
          overflow: hidden;
        }
        :host(.fullscreen) {
          width: 100%;
          height: 100%;
          position: fixed;
          top: 0;
          left: 0;
        }
        :host(.fullscreen) svg {
          width: 345px;
          height: 345px;
        }
        @media (prefers-reduced-motion: reduce) {
          .joomla-spinner {
            animation: none !important;
          }
        }
        @keyframes joomla-spinner {
          0%, 28%, to {
            opacity: 0.30;
          }
          20% {
            opacity: 1;
          }
        }
        .joomla-spinner {
          animation: joomla-spinner 1.6s infinite cubic-bezier(0, 0.15, 1, 0.75);
        }
      </style>
      <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 1001 992" width="${this.size}" height="${this.size}">
        <path d="M92 987c394,17 92,-566 486,-549 -2,48 5,95 3,143 -392,-66 -97,471 -489,406z" style="fill:#FFAA00;animation-delay:-1.8s;" class="joomla-spinner"></path>
        <path d="M92 987c374,-125 -119,-561 256,-685 15,45 39,85 54,130 -389,79 79,476 -310,555z" style="fill:white;animation-delay:-1.8s;" class="joomla-spinner"></path>
        <path d="M88 984c306,-249 -306,-484 0,-734 31,37 68,73 98,110 -336,212 238,412 -98,624z" style="fill:#FFAA00;animation-delay:-1.8s;" class="joomla-spinner"></path>

        <path d="M661 470c21,1 38,19 37,40 -1,21 -19,37 -40,36 -21,-1 -37,-19 -36,-40 1,-21 19,-37 39,-36z" style="fill:#FFAA00;animation-delay:-1.4s;" class="joomla-spinner"></path>
        <path d="M445 301c21,-5 41,8 46,28 5,20 -7,41 -27,46 -20,5 -41,-7 -46,-27 -5,-21 7,-41 27,-47z" style="fill:white;animation-delay:-1.4s;" class="joomla-spinner"></path>
        <path d="M173 219c16,-13 40,-10 53,6 13,17 10,41 -7,54 -16,13 -40,10 -53,-7 -13,-17 -10,-40 7,-53z" style="fill:#FFAA00;animation-delay:-1.4s;" class="joomla-spinner"></path>

        <path d="M788 453c27,1 48,24 46,52 -1,27 -24,48 -51,47 -27,-2 -48,-25 -47,-52 1,-27 24,-48 52,-47z" style="fill:#FFAA00;animation-delay:-1.0s;" class="joomla-spinner"></path>
        <path d="M561 249c26,-7 53,9 60,35 7,27 -9,54 -36,60 -26,7 -53,-9 -60,-35 -6,-27 10,-54 36,-60z" style="fill:white;animation-delay:-1.0s;" class="joomla-spinner"></path>
        <path d="M258 124c21,-17 52,-13 69,9 17,21 13,52 -9,69 -21,17 -52,13 -69,-9 -17,-21 -13,-52 9,-69z" style="fill:#FFAA00;animation-delay:-1.0s;" class="joomla-spinner"></path>

        <path d="M942 439c34,2 61,31 59,66 -1,35 -31,61 -65,60 -35,-2 -61,-31 -60,-66 2,-35 31,-61 66,-60z" style="fill:#FFAA00;animation-delay:-0.6s;" class="joomla-spinner"></path>
        <path d="M705 190c33,-8 67,12 76,46 8,33 -12,68 -46,76 -33,9 -67,-12 -76,-45 -8,-34 12,-68 46,-77z" style="fill:white;animation-delay:-0.6s;" class="joomla-spinner"></path>
        <path d="M366 13c28,-21 67,-16 88,11 21,28 16,67 -11,88 -27,22 -67,17 -88,-11 -21,-27 -16,-67 11,-88z" style="fill:#FFAA00;animation-delay:-0.6s;" class="joomla-spinner"></path>
      </svg>`;
    this.shadowRoot.appendChild(template.content.cloneNode(true));
  }
  connectedCallback() {
    this.style.backgroundColor = this.color ? this.color : getColorScheme();
    darkModeMediaQuery.addEventListener('change', this.systemQuery);
    if (!this.inline) {
      this.classList.add('fullscreen');
    }
  }
  disconnectedCallback() {
    darkModeMediaQuery.removeEventListener('change', this.systemQuery);
  }
  attributeChangedCallback(attr, oldValue, newValue) {
    switch (attr) {
      case 'color':
        if (newValue && newValue !== oldValue) {
          this.style.backgroundColor = newValue;
        }
        break;
      case 'size':
        if (newValue && newValue !== oldValue) {
          const svg = this.shadowRoot.querySelector('svg');
          svg.setAttribute('width', newValue);
          svg.setAttribute('height', newValue);
        }
        break;
      case 'inline':
        if (this.hasAttribute('inline')) {
          this.classList.remove('fullscreen');
        } else {
          this.classList.add('fullscreen');
        }
        break;
    }
  }
  systemQuery(event) {
    if (!notForced() || this.color) return;
    const color = event.matches === true ? darkColor : lightColor;
    if (this.style.backgroundColor !== color) {
      this.style.backgroundColor = color;
    }
  }
}
window.customElements.define('joomla-core-loader', JoomlaCoreLoader);
